
// __________________________________________________________
//
//                          ntinfo.c
//                Windows NT System Info V1.00
//                11-15-1998 Sven B. Schreiber
//                       sbs@orgon.com
// __________________________________________________________

#include "ntinfo.h"

// =================================================================
// DISCLAIMER
// =================================================================

/*

This software is provided "as is" and any express or implied
warranties, including, but not limited to, the implied warranties of
merchantibility and fitness for a particular purpose are disclaimed.
In no event shall the author Sven B. Schreiber be liable for any
direct, indirect, incidental, special, exemplary, or consequential
damages (including, but not limited to, procurement of substitute
goods or services; loss of use, data, or profits; or business
interruption) however caused and on any theory of liability,
whether in contract, strict liability, or tort (including negligence
or otherwise) arising in any way out of the use of this software,
even if advised of the possibility of such damage.

*/

// =================================================================
// REVISION HISTORY
// =================================================================

/*

11-15-1998 V1.00 Original version (SBS).

*/

// =================================================================
// GLOBAL VARIABLES
// =================================================================

DWORD gdLine      =       80;
DWORD gdIdProcess = MAXDWORD;

// =================================================================
// GLOBAL STRINGS
// =================================================================

TBYTE atArguments        [] =
  T("{ <option> { [<parameter>] } }");

TBYTE atOptionPrefixes   [] =
  T("-/");

TBYTE atOptionInfo       [] =
  T("\r\n\0")
  T("Options:  \0")
  T("\r\n\"this\" is the current process, ") S(MAIN_FILENAME) T(".")
  T("\r\nTo exclude entries, specify \"<option> * <entry list>\".")
  T("\r\nUnambiguous abbreviations are allowed.")
  T("\r\nValid option prefixes:\0.\r\n");

// -----------------------------------------------------------------

TBYTE atOptionInvalid    [] =
  T("\r\n\\\\ The command option \"%s\" is invalid.");

TBYTE atOptionAmbiguous  [] =
  T("\r\n\\\\ The command option \"%s\" is ambiguous.");

TBYTE atOptionHelp       [] =
  T("\r\n\\\\ Please select one from the following list:")
  T("\r\n\0\\\\\0 %c%s\0\r\n");

TBYTE atParameterIgnored [] =
  T("\r\n\\\\ The command parameter \"%s\" was ignored.\r\n");

TBYTE atListError        [] =
  T("\r\n\\\\ Unable to retrieve the %s directory.");

// -----------------------------------------------------------------

TBYTE atCommand          [] =
  T("\r\n\0") S(MAIN_MODULE) T(" %c%s\0 %s\0 %lu\0")
  T("\r\n\0") T("=\0\r\n");

TBYTE atPageFile         [] =
  T("\r\n\0%lu\0 page file\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atLock             [] =
  T("\r\n\0%lu\0 lock\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atModule           [] =
  T("\r\n\0%lu\0 module\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atHandle           [] =
  T("\r\n\0%lu\0 handle\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atObject           [] =
  T("\r\n\0%lu\0 object type\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atProcess          [] =
  T("\r\n\0%lu\0 process\0\0es\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atThread           [] =
  T("\r\n\0%lu\0 thread\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atLookaside        [] =
  T("\r\n\0%lu\0 lookaside list\0\0s\0 (\0%lu\0 byte\0\0s\0)\r\n");

TBYTE atSummary          [] =
  T("\r\n\0%lu\0 item\0\0s\0 listed\r\n");

// -----------------------------------------------------------------

TBYTE atYes              [] = T("YES");
TBYTE atNo               [] = T("NO");
TBYTE atNewLine          [] = T("\r\n");
TBYTE atSpace            [] = T(" ");
TBYTE atDash             [] = T("-");

// -----------------------------------------------------------------

BYTE  abNull             [] =   "" ;
WORD  awNull             [] =  L"" ;
TBYTE atNull             [] = T("");

// =================================================================
// OPTION CONTROL TABLE
// =================================================================

DWORD WINAPI Basic     (PNTL_CMD_DATA pncd);
DWORD WINAPI Cpu       (PNTL_CMD_DATA pncd);
DWORD WINAPI Time      (PNTL_CMD_DATA pncd);
DWORD WINAPI Memory    (PNTL_CMD_DATA pncd);
DWORD WINAPI Cache     (PNTL_CMD_DATA pncd);
DWORD WINAPI PageFile  (PNTL_CMD_DATA pncd);
DWORD WINAPI Lock      (PNTL_CMD_DATA pncd);
DWORD WINAPI Module    (PNTL_CMD_DATA pncd);
DWORD WINAPI Handle    (PNTL_CMD_DATA pncd);
DWORD WINAPI Object    (PNTL_CMD_DATA pncd);
DWORD WINAPI Process   (PNTL_CMD_DATA pncd);
DWORD WINAPI Thread    (PNTL_CMD_DATA pncd);
DWORD WINAPI Lookaside (PNTL_CMD_DATA pncd);
DWORD WINAPI Help      (PNTL_CMD_DATA pncd);

// -----------------------------------------------------------------

NTL_CMD_OPTION anco [] =
    {
    T("basic"),     NULL,                                 Basic,
    T("cpu"),       NULL,                                 Cpu,
    T("time"),      NULL,                                 Time,
    T("memory"),    NULL,                                 Memory,
    T("cache"),     NULL,                                 Cache,
    T("pagefile"),  NULL,                                 PageFile,
    T("lock"),      T(" [*|all]"),                        Lock,
    T("module"),    T(" [*|all] { [this|<ID> ] }"),       Module,
    T("handle"),    T(" [*|all] { [this|<PID>] }"),       Handle,
    T("object"),    T(" [*|all] { [this|<PID>] }"),       Object,
    T("process"),   T(" [*|all] { [this|<PID>] }"),       Process,
    T("thread"),    T(" [*|all] { [this|<PID>|<TID>] }"), Thread,
    T("lookaside"), T(" [*|all] { [<tag pattern>] }"),    Lookaside,
    T("help"),      NULL,                                 Help,
    T("?"),         NULL,                                 Help,
    NULL,           NULL,                                 NULL,
    };

// -----------------------------------------------------------------

NTL_CMD_FLAG ancf [] =
    {
    T("*"),     NTL_CMD_FLAG_ALL,
    T("all"),   NTL_CMD_FLAG_ALL,
    T("this"),  NTL_CMD_FLAG_THIS,
    T("named"), NTL_CMD_FLAG_NAMED,
    NULL,       0,
    };

// =================================================================
// DISPLAY ROUTINES
// =================================================================

DWORD WINAPI DisplaySeparator (PTBYTE ptPattern,
                               DWORD  dSize,
                               BOOL   fLimit)
    {
    PTBYTE ptPattern1;
    DWORD  i, j, n;

    n = (fLimit && gdLine ? min (dSize, gdLine-1)
                          : dSize);

    ptPattern1 = ((ptPattern != NULL) && ptPattern [0] ? ptPattern
                                                       : atDash);

    for (i = j = 0; i < n; i++)
        {
        printf (T("%c"), ptPattern1 [j++]);
        if (!ptPattern1 [j]) j = 0;
        }
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI DisplayTest (PTBYTE ptFormat,
                          ...)
    {
    TBYTE atBuffer [NTL_WVSPRINTF];
    PVOID pArgs = &ptFormat + 1;

    wvsprintf (atBuffer, ptFormat, pArgs);
    return lstrlen (atBuffer);
    }

// -----------------------------------------------------------------

DWORD WINAPI DisplayText (PTBYTE ptText,
                          PDWORD pdOffset)
    {
    DWORD n = 0;

    if (pdOffset != NULL)
        {
        n = printf (ptText + *pdOffset);
        *pdOffset += (lstrlen (ptText + *pdOffset) + 1);
        }
    else
        {
        n = printf (ptText);
        }
    return n;
    }

// -----------------------------------------------------------------

void WINAPI DisplaySkip (PTBYTE ptText,
                         PDWORD pdOffset)
    {
    if (pdOffset != NULL)
        {
        *pdOffset += (lstrlen (ptText + *pdOffset) + 1);
        }
    return;
    }

// -----------------------------------------------------------------

void WINAPI DisplayCommand (PNTL_CMD_DATA pncd)
    {
    DWORD dOffset, dMask, i, n;

    dOffset = 0;
    DisplayText (atCommand, &dOffset);

    n = printf (atCommand+dOffset,
                pncd->tPrefix, pncd->pnco->ptName);

    DisplaySkip (atCommand, &dOffset);

    for (i = 0, dMask = ~0; pncd->pncfTable [i].ptName != NULL; i++)
        {
        if ((pncd->dFlags & dMask) & pncd->pncfTable [i].dMask)
            {
            n += printf (atCommand+dOffset,
                         pncd->pncfTable [i].ptName);

            dMask &= ~pncd->pncfTable [i].dMask;
            }
        }
    DisplaySkip (atCommand, &dOffset);

    for (i = 0; i < pncd->dCount; i++)
        {
        if (pncd->ancp [i].fNumeric)
            {
            n += printf (atCommand+dOffset,
                         pncd->ancp [i].dValue);
            }
        }
    DisplaySkip (atCommand, &dOffset);
    DisplayText (atCommand, &dOffset);

    DisplaySeparator (atCommand+dOffset, n, TRUE);

    DisplaySkip (atCommand, &dOffset);
    DisplayText (atCommand, &dOffset);
    return;
    }

// -----------------------------------------------------------------

void WINAPI DisplayHeader (PTBYTE ptText,
                           DWORD  dItems,
                           DWORD  dBytes)
    {
    DWORD dOffset;

    dOffset = 0;
    DisplayText (ptText, &dOffset);

    printf (ptText+dOffset, dItems);
    DisplaySkip (ptText, &dOffset);
    DisplayText (ptText, &dOffset);

    if (dItems == 1)
        {
        DisplayText (ptText, &dOffset);
        DisplaySkip (ptText, &dOffset);
        }
    else
        {
        DisplaySkip (ptText, &dOffset);
        DisplayText (ptText, &dOffset);
        }
    DisplayText (ptText, &dOffset);

    printf (ptText+dOffset, dBytes);
    DisplaySkip (ptText, &dOffset);
    DisplayText (ptText, &dOffset);

    if (dBytes == 1)
        {
        DisplayText (ptText, &dOffset);
        DisplaySkip (ptText, &dOffset);
        }
    else
        {
        DisplaySkip (ptText, &dOffset);
        DisplayText (ptText, &dOffset);
        }
    DisplayText (ptText, &dOffset);
    return;
    }

// -----------------------------------------------------------------

void WINAPI DisplaySummary (PTBYTE ptText,
                            DWORD  dItems)
    {
    DWORD dOffset;

    dOffset = 0;
    DisplayText (ptText, &dOffset);

    printf (ptText+dOffset, dItems);
    DisplaySkip (ptText, &dOffset);
    DisplayText (ptText, &dOffset);

    if (dItems == 1)
        {
        DisplayText (ptText, &dOffset);
        DisplaySkip (ptText, &dOffset);
        }
    else
        {
        DisplaySkip (ptText, &dOffset);
        DisplayText (ptText, &dOffset);
        }
    DisplayText (ptText, &dOffset);
    return;
    }

// =================================================================
// COMMAND LINE MANAGEMENT
// =================================================================

BOOL CALLBACK CommandCallback (DWORD          dEvent,
                               TBYTE          tPrefix,
                               PTBYTE         ptData,
                               PNTINFO_CLIENT pnc)
    {
    BOOL fMore = TRUE;

    switch (dEvent)
        {
        case NTL_CMD_CALLBACK_OVERFLOW:
            {
            printf (atParameterIgnored, ptData);
            break;
            }
        case NTL_CMD_CALLBACK_INVALID:
            {
            printf (atOptionInvalid, ptData);
            break;
            }
        case NTL_CMD_CALLBACK_AMBIGUOUS:
            {
            printf (atOptionAmbiguous, ptData);
            break;
            }
        case NTL_CMD_CALLBACK_ENUMERATE:
            {
            if (ptData != NULL)
                {
                if (!pnc->dOffset)
                    {
                    DisplayText (atOptionHelp, &pnc->dOffset);

                    pnc->dNewLine = pnc->dOffset;

                    pnc->dColumn  =
                        DisplayText (atOptionHelp, &pnc->dOffset);
                    }
                if (pnc->dColumn +
                    DisplayTest (atOptionHelp + pnc->dOffset,
                                 tPrefix, ptData)
                    >= gdLine)
                    {
                    printf (atNewLine);

                    pnc->dOffset = pnc->dNewLine;

                    pnc->dColumn =
                        DisplayText (atOptionHelp, &pnc->dOffset);
                    }
                pnc->dColumn += printf (atOptionHelp + pnc->dOffset,
                                        tPrefix, ptData);
                }
            else
                {
                if (pnc->dOffset)
                    {
                    DisplaySkip (atOptionHelp, &pnc->dOffset);
                    DisplayText (atOptionHelp, &pnc->dOffset);

                    pnc->dOffset = 0;
                    }
                }
            break;
            }
        }
    return fMore;
    }

// -----------------------------------------------------------------

DWORD WINAPI CommandParser (PNTL_CMD_OPTION pncoTable,
                            PNTL_CMD_FLAG   pncfTable,
                            PTBYTE          ptPrefixes,
                            PPTBYTE         pptData,
                            DWORD           dCount)
    {
    NTINFO_CLIENT nc = {0, 0, 0};

    return NtlCommandEvaluate (pncoTable, pncfTable, ptPrefixes,
                               pptData, dCount,
                               CommandCallback, &nc);
    }

// -----------------------------------------------------------------

BOOL WINAPI CommandInfo (PNTL_CMD_OPTION pncoTable,
                         TBYTE           tPrefix,
                         PTBYTE          ptPrefixes,
                         PTBYTE          ptInfo,
                         BOOL            fAlign)
    {
    DWORD dAlign, dOffset, dSpace, i, j, n;
    BOOL  fOk = FALSE;

    if (pncoTable != NULL)
        {
        for (dAlign = n = 0; pncoTable [n].ptName != NULL; n++)
            {
            if ((pncoTable [n].Handler != NULL) &&
                ((i = lstrlen (pncoTable [n].ptName)) > dAlign))
                {
                dAlign = i;
                }
            }
        if (dAlign)
            {
            for (dOffset = dSpace = i = 0; i < n; i++)
                {
                if (pncoTable [i].Handler != NULL)
                    {
                    if (!dOffset)
                        {
                        DisplayText (ptInfo, &dOffset);
                        dSpace = DisplayText (ptInfo, &dOffset);
                        }
                    else
                        {
                        for (j = 0; j < dSpace; j++)
                            {
                            printf (atSpace);
                            }
                        }
                    printf (T("%c"), tPrefix);
                    j = printf (pncoTable [i].ptName);

                    if (pncoTable [i].ptTail != NULL)
                        {
                        if (fAlign)
                            {
                            for (; j < dAlign; j++)
                                {
                                printf (atSpace);
                                }
                            }
                        printf (pncoTable [i].ptTail);
                        }
                    printf (atNewLine);
                    }
                }
            DisplayText (ptInfo, &dOffset);
            for (i = 0; ptPrefixes [i]; i++)
                {
                printf (T(" '%c'"), ptPrefixes [i]);
                }
            DisplayText (ptInfo, &dOffset);
            fOk = TRUE;
            }
        }
    return fOk;
    }

// =================================================================
// SYSTEM INFO HANDLERS
// =================================================================

BOOL WINAPI TestIds (PNTL_CMD_DATA pncd,
                     DWORD         dIdThis,
                     DWORD         dCount,
                     ...)
    {
    DWORD  i, j;
    BOOL   fAll;
    PDWORD pdIds = &dCount + 1;
    BOOL   fOk   = FALSE;

    fAll = (pncd->dCount == 0                ) ||
           (pncd->dFlags &  NTL_CMD_FLAG_ALL );

    fOk  = (pncd->dCount == 1                ) &&
           (pncd->dFlags &  NTL_CMD_FLAG_THIS) &&
           (dIdThis      == MAXDWORD         );

    if ((pncd->dFlags & NTL_CMD_FLAG_THIS) && (dIdThis != MAXDWORD))
        {
        for (j = 0; (!fOk) && (j < dCount); j++)
            {
            fOk = (pdIds [j] == dIdThis);
            }
        }
    for (i = 0; (!fOk) && (i < pncd->dCount); i++)
        {
        if (pncd->ancp [i].fNumeric)
            {
            for (j = 0; (!fOk) && (j < dCount); j++)
                {
                fOk = (pdIds [j] == pncd->ancp [i].dValue);
                }
            }
        }
    return (fAll ? !fOk : fOk);
    }

// -----------------------------------------------------------------

BOOL WINAPI TestTag (PNTL_CMD_DATA pncd,
                     DWORD         dTag)
    {
    DWORD i, j;
    BOOL  fAll;
    BOOL  fOk   = FALSE;

    fAll = (pncd->dCount == 0                ) ||
           (pncd->dFlags &  NTL_CMD_FLAG_ALL );

    for (i = 0; (!fOk) && (i < pncd->dCount); i++)
        {
        for (j = 0; (!fOk) && (j < 4); j++)
            {
            if ((!pncd->ancp [i].ptValue [j])
                ||
                ((pncd->ancp [i].ptValue [j] != '?') &&
                 (pncd->ancp [i].ptValue [j] !=
                      ((PBYTE) &dTag) [j])))
                {
                break;
                }
            }
        fOk = (!pncd->ancp [i].ptValue [j]);
        }
    return (fAll ? !fOk : fOk);
    }

// -----------------------------------------------------------------

DWORD WINAPI Basic (PNTL_CMD_DATA pncd)
    {
    SYSTEM_CONFIGURATION_INFORMATION    sci;
    SYSTEM_BASIC_INFORMATION            sbi;
    SYSTEM_DPC_INFORMATION              sdpc;
    SYSTEM_GLOBAL_FLAG                  sgf;
    SYSTEM_CRASH_DUMP_INFORMATION       scdi;
    SYSTEM_CRASH_DUMP_STATE_INFORMATION scdsi;
    SYSTEM_DEBUGGER_INFORMATION         sdi;
    DWORD                               dSize;

    DisplayCommand (pncd);

    dSize = SYSTEM_CONFIGURATION_INFORMATION_;
    if (NtlInfoRead (&sci, &dSize, NTL_INFO_SYSTEM,
                     SystemConfigurationInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nGlobal I/O configuration:     ")
                T("%lu HD, %lu FD, %lu CD, ")
                T("%lu Tape, %lu COM, %lu LPT"),
                sci.dDiskCount,
                sci.dFloppyCount,
                sci.dCDRomCount,
                sci.dTapeCount,
                sci.dSerialCount,
                sci.dParallelCount);
        }
    dSize = SYSTEM_BASIC_INFORMATION_;
    if (NtlInfoRead (&sbi, &dSize, NTL_INFO_SYSTEM,
                     SystemBasicInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nPhysical address range:       ")
                T("0x%08lX..0x%08lX")
                T("\r\nApplication address range:    ")
                T("0x%08lX..0x%08lX")
                T("\r\nMaximum time increment:       %lu (%lu Hz)"),
                NtlPagesToBytes (sbi.dMmLowestPhysicalPage),
                NtlPagesToBytes (sbi.dMmHighestPhysicalPage),
                sbi.pLowestUserAddress,
                sbi.pMmHighestUserAddress,
                sbi.dKeMaximumIncrement,
                (TICKS_PER_SECOND / sbi.dKeMaximumIncrement) +
                (TICKS_PER_SECOND % sbi.dKeMaximumIncrement
                 >= sbi.dKeMaximumIncrement >> 1 ? 1 : 0));
        }
    dSize = SYSTEM_DPC_INFORMATION_;
    if (NtlInfoRead (&sdpc, &dSize, NTL_INFO_SYSTEM,
                     SystemDpcInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nMaximum DPC queue depth:      %lu")
                T("\r\nMinimum DPC rate:             %lu")
                T("\r\nIdeal DPC rate:               %lu")
                T("\r\nAdjust DPC threshold:         %lu"),
                sdpc.dKiMaximumDpcQueueDepth,
                sdpc.dKiMinimumDpcRate,
                sdpc.dKiIdealDpcRate,
                sdpc.dKiAdjustDpcThreshold);
        }
    dSize = SYSTEM_GLOBAL_FLAG_;
    if (NtlInfoRead (&sgf, &dSize, NTL_INFO_SYSTEM,
                     SystemGlobalFlag, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nGlobal flag:                  0x%08lX"),
                sgf.dNtGlobalFlag);
        }
    dSize = SYSTEM_CRASH_DUMP_INFORMATION_;
    if (NtlInfoRead (&scdi, &dSize, NTL_INFO_SYSTEM,
                     SystemCrashDumpInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nCrash dump section handle:    0x%08lX"),
                scdi.hMmCrashDumpSection);
        }
    dSize = SYSTEM_CRASH_DUMP_STATE_INFORMATION_;
    if (NtlInfoRead (&scdsi, &dSize, NTL_INFO_SYSTEM,
                     SystemCrashDumpStateInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nCrash dump enabled:           %s"),
                (scdsi.fCrashDumpEnabled ? atYes : atNo));
        }
    dSize = SYSTEM_DEBUGGER_INFORMATION_;
    if (NtlInfoRead (&sdi, &dSize, NTL_INFO_SYSTEM,
                     SystemDebuggerInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nKernel debugger enabled:      %s")
                T("\r\nKernel debugger not present:  %s"),
                (sdi.bKdDebuggerEnabled    ? atYes : atNo),
                (sdi.bKdDebuggerNotPresent ? atYes : atNo));
        }
    printf (atNewLine);
    return 0;
    }

// -----------------------------------------------------------------

DWORD WINAPI Cpu (PNTL_CMD_DATA pncd)
    {
    SYSTEM_PROCESSOR_INFORMATION     spi;
    SYSTEM_BASIC_INFORMATION         sbi;
    SYSTEM_THREAD_SWITCH_INFORMATION stsi;
    SYSTEM_TIME_INFORMATION          sti;
    PSYSTEM_PROCESSOR_COUNTERS       pspc;
    NTL_TABLE                        nt;
    TIME_FIELDS                      tf, tfT, tfP, tfU, tfI, tfD;
    PTBYTE                           ptProcessorArchitecture;
    QWORD                            qTime;
    DWORD                            dSize, i;

    DisplayCommand (pncd);

    dSize = SYSTEM_PROCESSOR_INFORMATION_;
    if (NtlInfoRead (&spi, &dSize, NTL_INFO_SYSTEM,
                     SystemProcessorInformation, NULL)
        == STATUS_SUCCESS)
        {
        switch (spi.wKeProcessorArchitecture)
            {
            case PROCESSOR_ARCHITECTURE_INTEL:
                ptProcessorArchitecture = T("x86");
                break;

            case PROCESSOR_ARCHITECTURE_MIPS:
                ptProcessorArchitecture = T("MIPS");
                break;

            case PROCESSOR_ARCHITECTURE_ALPHA:
                ptProcessorArchitecture = T("ALPHA");
                break;

            case PROCESSOR_ARCHITECTURE_PPC:
                ptProcessorArchitecture = T("PPC");
                break;

            default:
                ptProcessorArchitecture = T("Unknown");
                break;
            }
        printf (T("\r\nProcessor architecture:       %u (%s)")
                T("\r\nProcessor level:              %u")
                T("\r\nProcessor revision:           %u.%u")
                T("\r\nProcessor feature bits:       0x%08lX"),
                spi.wKeProcessorArchitecture,
                ptProcessorArchitecture,
                spi.wKeProcessorLevel,
                HIBYTE (spi.wKeProcessorRevision),
                LOBYTE (spi.wKeProcessorRevision),
                spi.dKeFeatureBits);
        }
    dSize = SYSTEM_BASIC_INFORMATION_;
    if (NtlInfoRead (&sbi, &dSize, NTL_INFO_SYSTEM,
                     SystemBasicInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nNumber of processors:         %lu")
                T("\r\nActive processor mask:        0x%08lX"),
                sbi.bKeNumberProcessors,
                sbi.dKeActiveProcessors);
        }
    dSize = SYSTEM_THREAD_SWITCH_INFORMATION_;
    if (NtlInfoRead (&stsi, &dSize, NTL_INFO_SYSTEM,
                     SystemThreadSwitchInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nTotal context switches:       %lu"),
                stsi.dTotalContextSwitches);
        }
    dSize = SYSTEM_TIME_INFORMATION_;
    if (NtlInfoRead (&sti, &dSize, NTL_INFO_SYSTEM,
                     SystemTimeInformation, NULL)
        == STATUS_SUCCESS)
        {
        NtlTimeInterval
            (&sti.qKeBootTime, &sti.qKeSystemTime, &qTime, 1);

        NtlTimeUnpack (&qTime, &tf);

        printf (T("\r\nSystem up time:               ")
                T("%02u:%02u:%02u"),
                tf.Hour, tf.Minute, tf.Second);
        }
    if (NtlTableCpu (&nt) == STATUS_SUCCESS)
        {
        pspc = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            NtlTimeUnpack (&pspc->qTotalTime,     &tfT);
            NtlTimeUnpack (&pspc->qProcessorTime, &tfP);
            NtlTimeUnpack (&pspc->qUserTime,      &tfU);
            NtlTimeUnpack (&pspc->qInterruptTime, &tfI);
            NtlTimeUnpack (&pspc->qDpcTime,       &tfD);

            printf (T("\r\n")
                    T("\r\nCPU #%02lu total time:           ")
                    T("%02u:%02u:%02u")
                    T("\r\nCPU #%02lu processor time:       ")
                    T("%02u:%02u:%02u")
                    T("\r\nCPU #%02lu user mode time:       ")
                    T("%02u:%02u:%02u")
                    T("\r\nCPU #%02lu interrupt time:       ")
                    T("%02u:%02u:%02u")
                    T("\r\nCPU #%02lu DPC time:             ")
                    T("%02u:%02u:%02u")
                    T("\r\nCPU #%02lu interrupt count:      %lu"),
                    i+1, tfT.Hour, tfT.Minute, tfT.Second,
                    i+1, tfP.Hour, tfP.Minute, tfP.Second,
                    i+1, tfU.Hour, tfU.Minute, tfU.Second,
                    i+1, tfI.Hour, tfI.Minute, tfI.Second,
                    i+1, tfD.Hour, tfD.Minute, tfD.Second,
                    i+1, pspc->dInterruptCount);

            pspc = NtlTableNext (&nt, pspc);
            }
        NtlTableUnload (&nt);
        }
    printf (atNewLine);
    return 0;
    }

// -----------------------------------------------------------------

DWORD WINAPI Time (PNTL_CMD_DATA pncd)
    {
    SYSTEM_BASIC_INFORMATION           sbi;
    SYSTEM_TIME_INFORMATION            sti;
    SYSTEM_TIME_ADJUSTMENT_INFORMATION stai;
    SYSTEM_TIME_ZONE_INFORMATION       stzi;
    SYSTEM_PERFORMANCE_INFORMATION     spi;
    TIME_FIELDS                        tf;
    QWORD                              qBias, qTime;
    DWORD                              dSize;

    DisplayCommand (pncd);

    dSize = SYSTEM_TIME_ADJUSTMENT_INFORMATION_;
    if (NtlInfoRead (&stai, &dSize, NTL_INFO_SYSTEM,
                     SystemTimeAdjustmentInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nSystem time adjustment:       "));

        if (stai.bKeTimeSynchronization)
            {
            printf (T("Disabled"));
            }
        else
            {
            printf (T("Adjustment=%lu, Increment=%lu"),
                    stai.TimeAdjustmentParams.dKeTimeAdjustment,
                    stai.TimeAdjustmentParams.dKeMaximumIncrement);
            }
        }
    dSize = SYSTEM_BASIC_INFORMATION_;
    if (NtlInfoRead (&sbi, &dSize, NTL_INFO_SYSTEM,
                     SystemBasicInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nMaximum time increment:       %lu"),
                sbi.dKeMaximumIncrement);
        }
    dSize = SYSTEM_TIME_ZONE_INFORMATION_;
    if (NtlInfoRead (&stzi, &dSize, NTL_INFO_SYSTEM,
                     SystemTimeZoneInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nStandard -> Daylight:         ")
                T("%02u-%02u %02u:%02u:%02u [%ls]")
                T("\r\nDaylight -> Standard:         ")
                T("%02u-%02u %02u:%02u:%02u [%ls]")
                T("\r\nStandard UTC bias (minutes):  %li")
                T("\r\nDaylight UTC bias (minutes):  %li"),
                stzi.tfDaylightDate.Month,
                stzi.tfDaylightDate.Day,
                stzi.tfDaylightDate.Hour,
                stzi.tfDaylightDate.Minute,
                stzi.tfDaylightDate.Second,
                stzi.awDaylightName,
                stzi.tfStandardDate.Month,
                stzi.tfStandardDate.Day,
                stzi.tfStandardDate.Hour,
                stzi.tfStandardDate.Minute,
                stzi.tfStandardDate.Second,
                stzi.awStandardName,
                stzi.lBias + stzi.lStandardBias,
                stzi.lBias + stzi.lDaylightBias);
        }
    dSize = SYSTEM_TIME_INFORMATION_;
    if (NtlInfoRead (&sti, &dSize, NTL_INFO_SYSTEM,
                     SystemTimeInformation, NULL)
        == STATUS_SUCCESS)
        {
        qBias = sti.dExpTimeZoneBias;
        NtlTimeUnits (&qBias, TICKS_PER_MINUTE);

        printf (T("\r\nCurrent  UTC bias (minutes):  %li"),
                qBias.LowPart);

        NtlTimeUnpack (&sti.qKeBootTime, &tf);

        printf (T("\r\nSystem startup time (UTC):    ")
                T("%02u-%02u-%04u %02u:%02u:%02u"),
                tf.Month, tf.Day,    tf.Year,
                tf.Hour,  tf.Minute, tf.Second);

        NtlTimeInterval
            (&sti.dExpTimeZoneBias, &sti.qKeBootTime, &qTime, 1);

        NtlTimeUnpack (&qTime, &tf);

        printf (T("\r\nSystem startup time (local):  ")
                T("%02u-%02u-%04u %02u:%02u:%02u"),
                tf.Month, tf.Day,    tf.Year,
                tf.Hour,  tf.Minute, tf.Second);

        NtlTimeUnpack (&sti.qKeSystemTime, &tf);

        printf (T("\r\nCurrent system time (UTC):    ")
                T("%02u-%02u-%04u %02u:%02u:%02u"),
                tf.Month, tf.Day,    tf.Year,
                tf.Hour,  tf.Minute, tf.Second);

        NtlTimeInterval
            (&sti.dExpTimeZoneBias, &sti.qKeSystemTime, &qTime, 1);

        NtlTimeUnpack (&qTime, &tf);

        printf (T("\r\nCurrent system time (local):  ")
                T("%02u-%02u-%04u %02u:%02u:%02u"),
                tf.Month, tf.Day,    tf.Year,
                tf.Hour,  tf.Minute, tf.Second);

        NtlTimeInterval
            (&sti.qKeBootTime, &sti.qKeSystemTime, &qTime, 1);

        NtlTimeUnpack (&qTime, &tf);

        printf (T("\r\nSystem up time:               ")
                T("%02u:%02u:%02u"),
                tf.Hour, tf.Minute, tf.Second);
        }
    dSize = SYSTEM_PERFORMANCE_INFORMATION_;
    if (NtlInfoRead (&spi, &dSize, NTL_INFO_SYSTEM,
                     SystemPerformanceInformation, NULL)
        == STATUS_SUCCESS)
        {
        NtlTimeUnpack (&spi.qIdleTime, &tf);

        printf (T("\r\nSystem idle time:             ")
                T("%02u:%02u:%02u"),
                tf.Hour, tf.Minute, tf.Second);
        }
    printf (atNewLine);
    return 0;
    }

// -----------------------------------------------------------------

DWORD WINAPI Memory (PNTL_CMD_DATA pncd)
    {
    SYSTEM_BASIC_INFORMATION          sbi;
    SYSTEM_PERFORMANCE_INFORMATION    spi;
    SYSTEM_REGISTRY_QUOTA_INFORMATION srqi;
    DWORD                             dSize;

    DisplayCommand (pncd);

    dSize = SYSTEM_BASIC_INFORMATION_;
    if (NtlInfoRead (&sbi, &dSize, NTL_INFO_SYSTEM,
                     SystemBasicInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nPage size:                    %10lu %7lu K")
                T("\r\nAllocation granularity:       %10lu %7lu K")
                T("\r\nPhysical memory total:        %10lu %7lu K"),
                                (sbi.dPageSize),
                NtlBytesToKb    (sbi.dPageSize),
                                (sbi.dAllocationGranularity),
                NtlBytesToKb    (sbi.dAllocationGranularity),
                NtlPagesToBytes (sbi.dMmNumberOfPhysicalPages),
                NtlPagesToKb    (sbi.dMmNumberOfPhysicalPages));
        }
    dSize = SYSTEM_PERFORMANCE_INFORMATION_;
    if (NtlInfoRead (&spi, &dSize, NTL_INFO_SYSTEM,
                     SystemPerformanceInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nPhysical memory available:    %10lu %7lu K")
                T("\r\nCommit charge total:          %10lu %7lu K")
                T("\r\nCommit charge limit:          %10lu %7lu K")
                T("\r\nCommit charge peak:           %10lu %7lu K")
                T("\r\nPool memory paged:            %10lu %7lu K")
                T("\r\nPool memory non-paged:        %10lu %7lu K"),
                NtlPagesToBytes (spi.dMmAvailablePages     ),
                NtlPagesToKb    (spi.dMmAvailablePages     ),
                NtlPagesToBytes (spi.dMmTotalCommittedPages),
                NtlPagesToKb    (spi.dMmTotalCommittedPages),
                NtlPagesToBytes (spi.dMmTotalCommitLimit   ),
                NtlPagesToKb    (spi.dMmTotalCommitLimit   ),
                NtlPagesToBytes (spi.dMmPeakCommitLimit    ),
                NtlPagesToKb    (spi.dMmPeakCommitLimit    ),
                NtlPagesToBytes (spi.dPoolPaged            ),
                NtlPagesToKb    (spi.dPoolPaged            ),
                NtlPagesToBytes (spi.dPoolNonPaged         ),
                NtlPagesToKb    (spi.dPoolNonPaged         ),
                NtlPagesToBytes (spi.dMmPagedPoolPage      ));
        }
    dSize = SYSTEM_REGISTRY_QUOTA_INFORMATION_;
    if (NtlInfoRead (&srqi, &dSize, NTL_INFO_SYSTEM,
                     SystemRegistryQuotaInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nRegistry size limit:          %10lu %7lu K")
                T("\r\nCurrent registry size:        %10lu %7lu K")
                T("\r\nPaged pool size:              %10lu %7lu K"),
                             (srqi.dCmpGlobalQuota),
                NtlBytesToKb (srqi.dCmpGlobalQuota),
                             (srqi.dCmpGlobalQuotaUsed),
                NtlBytesToKb (srqi.dCmpGlobalQuotaUsed),
                             (srqi.dMmSizeOfPagedPoolInBytes),
                NtlBytesToKb (srqi.dMmSizeOfPagedPoolInBytes));
        }
    printf (atNewLine);
    return 0;
    }

// -----------------------------------------------------------------

DWORD WINAPI Cache (PNTL_CMD_DATA pncd)
    {
    SYSTEM_CACHE_INFORMATION       sci;
    SYSTEM_PERFORMANCE_INFORMATION spi;
    DWORD                          dSize;

    DisplayCommand (pncd);

    dSize = SYSTEM_CACHE_INFORMATION_;
    if (NtlInfoRead (&sci, &dSize, NTL_INFO_SYSTEM,
                     SystemCacheInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nFile cache:                   ")
                T("%10lu %7lu K %4lu M")
                T("\r\nFile cache peak:              ")
                T("%10lu %7lu K %4lu M"),
                              sci.dFileCache,
                NtlBytesToKb (sci.dFileCache),
                NtlBytesToMb (sci.dFileCache),
                              sci.dFileCachePeak,
                NtlBytesToKb (sci.dFileCachePeak),
                NtlBytesToMb (sci.dFileCachePeak));
        }
    dSize = SYSTEM_PERFORMANCE_INFORMATION_;
    if (NtlInfoRead (&spi, &dSize, NTL_INFO_SYSTEM,
                     SystemPerformanceInformation, NULL)
        == STATUS_SUCCESS)
        {
        printf (T("\r\nSystem cache page:            0x%08lX\r\n")
                T("\r\nFast read no wait:            %10lu")
                T("\r\nFast read wait:               %10lu")
                T("\r\nFast read resource miss:      %10lu")
                T("\r\nFast read not possible:       %10lu\r\n")
                T("\r\nFast MDL read no wait:        %10lu")
                T("\r\nFast MDL read wait:           %10lu")
                T("\r\nFast MDL read resource miss:  %10lu")
                T("\r\nFast MDL read not possible:   %10lu\r\n")
                T("\r\nMap data no wait:             %10lu")
                T("\r\nMap data wait:                %10lu")
                T("\r\nMap data no wait miss:        %10lu")
                T("\r\nMap data wait miss:           %10lu\r\n"),
                NtlPagesToBytes (spi.dMmSystemCachePage),
                spi.dCcFastReadNoWait,
                spi.dCcFastReadWait,
                spi.dCcFastReadResourceMiss,
                spi.dCcFastReadNotPossible,
                spi.dCcFastMdlReadNoWait,
                spi.dCcFastMdlReadWait,
                spi.dCcFastMdlReadResourceMiss,
                spi.dCcFastMdlReadNotPossible,
                spi.dCcMapDataNoWait,
                spi.dCcMapDataWait,
                spi.dCcMapDataNoWaitMiss,
                spi.dCcMapDataWaitMiss);

        printf (T("\r\nPin mapped data count:        %10lu")
                T("\r\nPin read no wait:             %10lu")
                T("\r\nPin read wait:                %10lu")
                T("\r\nPin read no wait miss:        %10lu")
                T("\r\nPin read wait miss:           %10lu\r\n")
                T("\r\nCopy read no wait:            %10lu")
                T("\r\nCopy read wait:               %10lu")
                T("\r\nCopy read no wait miss:       %10lu")
                T("\r\nCopy read wait miss:          %10lu\r\n")
                T("\r\nMDL read no wait:             %10lu")
                T("\r\nMDL read wait:                %10lu")
                T("\r\nMDL read no wait miss:        %10lu")
                T("\r\nMDL read wait miss:           %10lu\r\n")
                T("\r\nRead ahead I/Os:              %10lu")
                T("\r\nLazy write I/Os:              %10lu")
                T("\r\nLazy write Pages:             %10lu")
                T("\r\nData flushes:                 %10lu")
                T("\r\nData pages:                   %10lu"),
                spi.dCcPinMappedDataCount,
                spi.dCcPinReadNoWait,
                spi.dCcPinReadWait,
                spi.dCcPinReadNoWaitMiss,
                spi.dCcPinReadWaitMiss,
                spi.dCcCopyReadNoWait,
                spi.dCcCopyReadWait,
                spi.dCcCopyReadNoWaitMiss,
                spi.dCcCopyReadWaitMiss,
                spi.dCcMdlReadNoWait,
                spi.dCcMdlReadWait,
                spi.dCcMdlReadNoWaitMiss,
                spi.dCcMdlReadWaitMiss,
                spi.dCcReadAheadIos,
                spi.dCcLazyWriteIos,
                spi.dCcLazyWritePages,
                spi.dCcDataFlushes,
                spi.dCcDataPages);
        }
    printf (atNewLine);
    return 0;
    }

// -----------------------------------------------------------------

DWORD WINAPI PageFile (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE                     nt;
    PSYSTEM_PAGE_FILE_INFORMATION pspfi;
    DWORD                         i;
    DWORD                         n = 0;

    DisplayCommand (pncd);

    if (NtlTablePageFile (&nt) == STATUS_SUCCESS)
        {
        DisplayHeader (atPageFile, nt.dCount, nt.dData);

        printf (T("\r\n")
                T("    Total    In use      Peak   Name\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        pspfi = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            printf (T("\r\n%7lu K %7lu K %7lu K   \"%ls\""),
                    NtlPagesToKb (pspfi->dTotal),
                    NtlPagesToKb (pspfi->dInUse),
                    NtlPagesToKb (pspfi->dPeak ),
                    pspfi->usName.Buffer);
            n++;

            pspfi = NtlTableNext (&nt, pspfi);
            }
        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("page file"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Lock (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE    nt;
    PSYSTEM_LOCK psl;
    DWORD        i;
    DWORD        n = 0;

    DisplayCommand (pncd);

    if (NtlTableLock (&nt) == STATUS_SUCCESS)
        {
        DisplayHeader (atLock, nt.dCount, nt.dData);

        printf (T("\r\n")
                T("   Lock        Lock         Active  ")
                T("Contention      Shared   Exclusive\r\n")
                T("List Entry     Owner         Count  ")
                T("     Count     Waiters     Waiters\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        psl = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            if ((pncd->dFlags & NTL_CMD_FLAG_ALL) ||
                psl->dActiveCount                 ||
                psl->dContentionCount             ||
                psl->dNumberOfSharedWaiters       ||
                psl->dNumberOfExclusiveWaiters)
                {
                printf (T("\r\n0x%08lX  0x%08lX  %10lu  ")
                        T("%10lu  %10lu  %10lu"),
                        psl->pEResource,
                        psl->dOwner,
                        psl->dActiveCount,
                        psl->dContentionCount,
                        psl->dNumberOfSharedWaiters,
                        psl->dNumberOfExclusiveWaiters);
                n++;
                }
            psl = NtlTableNext (&nt, psl);
            }
        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("lock"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Module (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE      nt;
    PSYSTEM_MODULE psm;
    DWORD          i;
    DWORD          n = 0;

    DisplayCommand (pncd);

    if (NtlTableModule (&nt, pncd->dFlags & NTL_CMD_FLAG_THIS)
        == STATUS_SUCCESS)
        {
        DisplayHeader (atModule, nt.dCount, nt.dData);

        printf (T("\r\n")
                T("  ID  Rank     Address       Size     Flags  ")
                T("Name\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        psm = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            if (TestIds (pncd, MAXDWORD, 1,
                         psm->wId))
                {
                printf (T("\r\n%4u  %4u  0x%08lX %10lu  %08lX  ")
                        T("\"%hs\""),
                        psm->wId,
                        psm->wRank,
                        psm->pAddress,
                        psm->dSize,
                        psm->dFlags,
                        psm->abName +
                        psm->wNameOffset);
                n++;
                }
            psm = NtlTableNext (&nt, psm);
            }
        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("module"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Handle (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE      nt;
    PSYSTEM_HANDLE psh;
    PTBYTE         ptName;
    DWORD          dSize, dMaxSize, i, j;
    DWORD          n = 0;

    DisplayCommand (pncd);

    if (NtlTableHandle (&nt) == STATUS_SUCCESS)
        {
        DisplayHeader (atHandle, nt.dCount, nt.dData);

        printf (T("\r\n")
                T("PID   Type            ")
                T("Flags    Value    ObjectBody   GrantedAcc\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        psh = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            if (TestIds (pncd, gdIdProcess, 1,
                         psh->dIdProcess))
                {
                printf (T("\r\n%3lu   "), psh->dIdProcess);

                ptName = NtlNameObjectType (psh->bObjectType,
                                            &dSize, &dMaxSize);

                if (ptName [0])
                    {
                    printf (ptName);
                    }
                else
                    {
                    printf (T("#%02lu"), psh->bObjectType);
                    dSize = 3;
                    }
                for (j = dSize; j < dMaxSize+3; j++)
                    {
                    printf (atSpace);
                    }
                printf (T("0x%04lX   0x%04X   0x%08lX   0x%08lX"),
                        psh->bFlags,
                        psh->wValue,
                        psh->pObject,
                        psh->GrantedAccess);
                n++;
                }
            psh = NtlTableNext (&nt, psh);
            }
        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("handle"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Object (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE                  nt, ntSub;
    PSYSTEM_OBJECT_INFORMATION psoi;
    PSYSTEM_OBJECT             pso;
    DWORD                      i, j;
    DWORD                      n = 0;

    DisplayCommand (pncd);

    if (NtlTableObject (&nt) == STATUS_SUCCESS)
        {
        DisplayHeader (atObject, nt.dCount, nt.dData);

        psoi = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            printf (T("\r\n")
                    T("Object type:         \"%ls\" (%lu)\r\n")
                    T("Object count:        %lu\r\n")
                    T("Handle count:        %lu\r\n")
                    T("Invalid attributes:  0x%08lX\r\n")
                    T("Valid access mask:   0x%08lX\r\n")
                    T("Generic mapping:     ")
                    T("R=0x%08lX W=0x%08lX X=0x%08lX A=0x%08lX\r\n")
                    T("Pool type:           \"%s\" (%lu)\r\n")
                    T("Security required:   %s\r\n"),
                    psoi->usTypeName.Buffer,
                    psoi->dObjectTypeCode,
                    psoi->dObjectCount,
                    psoi->dHandleCount,
                    psoi->dInvalidAttributes,
                    psoi->ValidAccessMask,
                    psoi->GenericMapping.GenericRead,
                    psoi->GenericMapping.GenericWrite,
                    psoi->GenericMapping.GenericExecute,
                    psoi->GenericMapping.GenericAll,
                    NtlNamePoolType (psoi->PoolType, NULL, NULL),
                    psoi->PoolType,
                    (psoi->bSecurityRequired ? atYes : atNo));

            printf (T("\r\n")
                    T("PID Ptr Hdl  Flags ")
                    T("ObjectBody  SecDescrpt")
                    T("  Name\r\n"));

            DisplaySeparator (NULL, 79, TRUE);

            pso = NtlTableObjectFirst (&ntSub, &nt, psoi);

            for (j = 0; j < ntSub.dCount; j++)
                {
                if ((pncd->dCount &&
                     TestIds (pncd, gdIdProcess, 1,
                              pso->dUniqueProcessId))
                    ||
                    ((!pncd->dCount) &&
                     (pso->usObjectName.Buffer != NULL)))
                    {
                    printf (T("\r\n%3lu %3lu %3lu  0x%02X  ")
                            T("0x%08lX  0x%08lX"),
                            pso->dUniqueProcessId,
                            pso->dPointerCount,
                            pso->dHandleCount,
                            pso->wObjectFlags,
                            pso->pObject,
                            pso->pSecurityDescriptor);

                    if (pso->usObjectName.Buffer != NULL)
                        {
                        printf (T("  \"%ls\""),
                                pso->usObjectName.Buffer);
                        }
                    n++;
                    }
                pso = NtlTableObjectNext (&ntSub, pso);
                }
            printf (atNewLine);
            psoi = NtlTableNext (&nt, psoi);
            }
        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("object type"));
        printf (atNewLine);
        }
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Process (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE                   nt;
    PSYSTEM_PROCESS_INFORMATION pspi;
    TIME_FIELDS                 tfC, tfK, tfU;
    QWORD                       qKernelTime, qUserTime;
    DWORD                       i;
    DWORD                       n = 0;

    DisplayCommand (pncd);

    if (NtlTableProcess (&nt) == STATUS_SUCCESS)
        {
        DisplayHeader (atProcess, nt.dCount, nt.dData);

        printf (T("\r\n")
                T("PID Org BP Th Hdls CommChrg WSetSize ")
                T("PFCount Start date and time Name\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        pspi = NtlTableFirst (&nt);

        NtlTimeReset (&qKernelTime);
        NtlTimeReset (&qUserTime);

        for (i = 0; i < nt.dCount; i++)
            {
            if (TestIds (pncd, gdIdProcess, 1,
                         pspi->dUniqueProcessId))
                {
                NtlTimeUnpack (&pspi->qCreateTime, &tfC);

                printf (T("\r\n")
                        T("%3lu %3lu %2lu %2lu %4lu %8lu %8lu ")
                        T("%7lu %02u-%02u-%04u %02u:%02u:%02u ")
                        T("\"%ls\""),
                        pspi->dUniqueProcessId,
                        pspi->dInheritedFromUniqueProcessId,
                        pspi->BasePriority,
                        pspi->dThreadCount,
                        pspi->dHandleCount,
                        pspi->dCommitCharge,
                        pspi->VmCounters.WorkingSetSize,
                        pspi->VmCounters.PageFaultCount,
                        tfC.Month, tfC.Day,    tfC.Year,
                        tfC.Hour,  tfC.Minute, tfC.Second,
                        (pspi->usName.Buffer != NULL
                         ? pspi->usName.Buffer
                         : L"Idle"));

                NtlTimeExtend (&qKernelTime,
                               &pspi->qKernelTime, 1);

                NtlTimeExtend (&qUserTime,
                               &pspi->qUserTime,   1);

                n++;
                }
            pspi = NtlTableNext (&nt, pspi);
            }
        NtlTimeUnpack (&qKernelTime, &tfK);
        NtlTimeUnpack (&qUserTime,   &tfU);

        printf (T("\r\n")
                T("\r\nTotal kernel time:  %02u:%02u:%02u")
                T("\r\nTotal user   time:  %02u:%02u:%02u"),
                tfK.Hour, tfK.Minute, tfK.Second,
                tfU.Hour, tfU.Minute, tfU.Second);

        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("process"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Thread (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE                   nt;
    PSYSTEM_PROCESS_INFORMATION pspi;
    TIME_FIELDS                 tfC, tfK, tfU;
    QWORD                       qKernelTime, qUserTime;
    DWORD                       dThreads, i, j;
    DWORD                       n = 0;

    DisplayCommand (pncd);

    if (NtlTableProcess (&nt) == STATUS_SUCCESS)
        {
        pspi = NtlTableFirst (&nt);

        for (dThreads = i = 0; i < nt.dCount; i++)
            {
            dThreads += pspi->dThreadCount;
            pspi = NtlTableNext (&nt, pspi);
            }
        DisplayHeader (atThread, dThreads, nt.dData);

        printf (T("\r\n")
                T("TID PID BP DP Switches TS WR ")
                T("Start Addr  ")
                T("Start date and time  ")
                T("Kernel   User\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        pspi = NtlTableFirst (&nt);

        NtlTimeReset (&qKernelTime);
        NtlTimeReset (&qUserTime);

        for (i = 0; i < nt.dCount; i++)
            {
            for (j = 0; j < pspi->dThreadCount; j++)
                {
                if (TestIds (pncd, gdIdProcess, 2,
                             pspi->ast [j].Cid.UniqueProcess,
                             pspi->ast [j].Cid.UniqueThread))
                    {
                    NtlTimeUnpack (&pspi->ast [j].qCreateTime,
                                   &tfC);

                    NtlTimeUnpack (&pspi->ast [j].qKernelTime,
                                   &tfK);

                    NtlTimeUnpack (&pspi->ast [j].qUserTime,
                                   &tfU);

                    printf (T("\r\n")
                            T("%3lu %3lu %2lu %2lu %8lu %2lu %2lu ")
                            T("0x%08lX  ")
                            T("%02u-%02u-%04u %02u:%02u:%02u  ")
                            T("%02u:%02u:%02u %02u:%02u:%02u"),
                            pspi->ast [j].Cid.UniqueThread,
                            pspi->ast [j].Cid.UniqueProcess,
                            pspi->ast [j].dBasePriority,
                            pspi->ast [j].dPriority,
                            pspi->ast [j].dContextSwitches,
                            pspi->ast [j].dThreadState,
                            pspi->ast [j].WaitReason,
                            pspi->ast [j].pStartAddress,
                            tfC.Month, tfC.Day,    tfC.Year,
                            tfC.Hour,  tfC.Minute, tfC.Second,
                            tfK.Hour,  tfK.Minute, tfK.Second,
                            tfU.Hour,  tfU.Minute, tfU.Second);

                    NtlTimeExtend (&qKernelTime,
                                   &pspi->ast [j].qKernelTime, 1);

                    NtlTimeExtend (&qUserTime,
                                   &pspi->ast [j].qUserTime,   1);

                    n++;
                    }
                }
            pspi = NtlTableNext (&nt, pspi);
            }
        NtlTimeUnpack (&qKernelTime, &tfK);
        NtlTimeUnpack (&qUserTime,   &tfU);

        printf (T("\r\n")
                T("\r\nTotal kernel time:  %02u:%02u:%02u")
                T("\r\nTotal user   time:  %02u:%02u:%02u"),
                tfK.Hour, tfK.Minute, tfK.Second,
                tfU.Hour, tfU.Minute, tfU.Second);

        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("thread"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Lookaside (PNTL_CMD_DATA pncd)
    {
    NTL_TABLE         nt;
    PSYSTEM_LOOKASIDE psla;
    DWORD             i;
    DWORD             n = 0;

    if (NtlTableLookaside (&nt) == STATUS_SUCCESS)
        {
        DisplayHeader (atLookaside, nt.dCount, nt.dData);

        printf (T("\r\n")
                T("Tag    Depth   Size    Allocs  ")
                T(" No Miss     Frees   No Miss    ")
                T("Pool Type\r\n"));

        DisplaySeparator (NULL, 79, TRUE);

        psla = NtlTableFirst (&nt);

        for (i = 0; i < nt.dCount; i++)
            {
            if (TestTag (pncd, psla->dPoolTag))
                {
                printf (T("\r\n")
                        T("\"%hc%hc%hc%hc\"  %4lu  %5lu  %8lu  ")
                        T("%8lu  %8lu  %8lu    ")
                        T("%s"),
                        ((PBYTE) &psla->dPoolTag) [0],
                        ((PBYTE) &psla->dPoolTag) [1],
                        ((PBYTE) &psla->dPoolTag) [2],
                        ((PBYTE) &psla->dPoolTag) [3],
                        psla->wDepth,
                        psla->dSize,
                        psla->dTotalAllocates,
                        psla->dAllocateNoMisses,
                        psla->dTotalFrees,
                        psla->dFreeNoMisses,
                        NtlNamePoolType (psla->PoolType,
                                         NULL, NULL));
                n++;
                }
            psla = NtlTableNext (&nt, psla);
            }
        NtlTableUnload (&nt);
        }
    else
        {
        printf (atListError, T("lookaside"));
        }
    printf (atNewLine);
    DisplaySummary (atSummary, n);
    return n;
    }

// -----------------------------------------------------------------

DWORD WINAPI Help (PNTL_CMD_DATA pncd)
    {
    TBYTE tPrefix = (pncd != NULL ? pncd->tPrefix
                                  : atOptionPrefixes [0]);

    printf (atUsage, atArguments);

    CommandInfo (anco, tPrefix, atOptionPrefixes,
                 atOptionInfo, TRUE);
    return 0;
    }

// =================================================================
// MAIN PROGRAM
// =================================================================

DWORD Main (DWORD argc, TBYTE *argv [], TBYTE *envp [])
    {
    CONSOLE_SCREEN_BUFFER_INFO csbi;
    DWORD                      dCodePage;

    gdIdProcess = GetCurrentProcessId ();

    dCodePage = GetConsoleOutputCP ();
    SetConsoleOutputCP (GetACP ());
    GetConsoleScreenBufferInfo (hStdOut, &csbi);
    gdLine = csbi.dwSize.X;

    printf (atAbout);

    if (argc < 2)
        {
        Help (NULL);
        }
    else
        {
        CommandParser (anco, ancf, atOptionPrefixes,
                       argv+1, argc-1);
        }
    SetConsoleOutputCP (dCodePage);
    return 0;
    }

// =================================================================
// END OF PROGRAM
// =================================================================
